<?php
namespace MikadoCore\PostTypes\Portfolio\Shortcodes;

use MikadoCore\Lib;

/**
 * Class PortfolioList
 * @package MikadoCore\PostTypes\Portfolio\Shortcodes
 */
class PortfolioList implements Lib\ShortcodeInterface {
	/**
	 * @var string
	 */
	private $base;

	public function __construct() {
		$this->base = 'mkdf_portfolio_list';

		add_action('vc_before_init', array($this, 'vcMap'));
	}

	/**
	 * Returns base for shortcode
	 * @return string
	 */
	public function getBase() {
		return $this->base;
	}

	/**
	 * Maps shortcode to Visual Composer
	 *
	 * @see vc_map
	 */
	public function vcMap() {
		if(function_exists('vc_map')) {

			$icons_array= array();
			if(mkd_core_theme_installed()) {
				$icons_array = \CortexMikadoIconCollections::get_instance()->getVCParamsArray();
			}

			vc_map( array(
					'name' => esc_html__('Portfolio List','mikado-cpt'),
					'base' => $this->getBase(),
					'category' => esc_html__('by MIKADO','mikado-cpt'),
					'icon' => 'icon-wpb-portfolio extended-custom-icon',
					'allowed_container_element' => 'vc_row',
					'params' => array(
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Portfolio List Template','mikado-cpt'),
							'param_name' => 'type',
							'value' => array(
								esc_html__('Standard','mikado-cpt') => 'standard',
								esc_html__('Gallery','mikado-cpt') => 'gallery',
								esc_html__('Gallery With Space','mikado-cpt') => 'gallery-with-space',
								esc_html__('Masonry','mikado-cpt') => 'masonry',
								esc_html__('Masonry With Space','mikado-cpt') => 'masonry-with-space',
								esc_html__('Pinterest','mikado-cpt') => 'pinterest',
								esc_html__('Pinterest With Space','mikado-cpt') => 'pinterest-with-space'
							),
							'admin_label' => true
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Title Tag','mikado-cpt'),
							'param_name' => 'title_tag',
							'value' => array(
								''   => '',
								'h2' => 'h2',
								'h3' => 'h3',
								'h4' => 'h4',
								'h5' => 'h5',
								'h6' => 'h6',
							)
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Image Proportions','mikado-cpt'),
							'param_name' => 'image_size',
							'value' => array(
								esc_html__('Original','mikado-cpt') => 'full',
								esc_html__('Square','mikado-cpt') => 'square',
								esc_html__('Landscape','mikado-cpt') => 'landscape',
								esc_html__('Portrait','mikado-cpt') => 'portrait'
							),
							'save_always' => true,
							'dependency' => array('element' => 'type', 'value' => array('standard', 'gallery','gallery-with-space'))
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Hover Type','mikado-cpt'),
							'param_name' => 'hover_type',
							'value' => array(
								esc_html__('Zoom Lightbox','mikado-cpt') => 'zoom-lightbox',
								esc_html__('Sweep to Left','mikado-cpt') => 'sweep-left',
								esc_html__('Trim Overlay','mikado-cpt') => 'trim',
								esc_html__('Zoom Out Simple','mikado-cpt') => 'zoom-out-simple',
							),
							'save_always' => true,
							'dependency' => array('element' => 'type', 'value' => array('gallery','gallery-with-space','masonry','masonry-with-space','pinterest','pinterest-with-space'))
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Show More','mikado-cpt'),
							'param_name' => 'show_more',
							'value' => array(
								esc_html__('Load More Button','mikado-cpt') => 'load_more',
								esc_html__('Infinite Scroll','mikado-cpt') => 'infinite_scroll',
								esc_html__('None','mikado-cpt') => 'none'
							),
							'save_always' => true,
							'description' => esc_html__('Default Value is Load More','mikado-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Order By','mikado-cpt'),
							'param_name' => 'order_by',
							'value' => array(
								esc_html__('Date','mikado-cpt') => 'date',
								esc_html__('Title','mikado-cpt') => 'title',
								esc_html__('Menu Order','mikado-cpt') => 'menu_order',
							),
							'save_always' => true,
							'group' => esc_html__('Query and Layout Options','mikado-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Order','mikado-cpt'),
							'param_name' => 'order',
							'value' => array(
								esc_html__('ASC','mikado-cpt') => 'ASC',
								esc_html__('DESC','mikado-cpt') => 'DESC',
							),
							'save_always' => true,
							'group' => esc_html__('Query and Layout Options','mikado-cpt')
						),
						array(
							'type' => 'textfield',
							'heading' => esc_html__('One-Category Portfolio List','mikado-cpt'),
							'param_name' => 'category',
							'description' => esc_html__('Enter one category slug (leave empty for showing all categories)','mikado-cpt'),
							'group' => esc_html__('Query and Layout Options','mikado-cpt')
						),
						array(
							'type' => 'textfield',
							'heading' => esc_html__('Number of Portfolios Per Page','mikado-cpt'),
							'param_name' => 'number',
							'value' => '-1',
							'description' => esc_html__('(enter -1 to show all)','mikado-cpt'),
							'group' => esc_html__('Query and Layout Options','mikado-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Number of Columns','mikado-cpt'),
							'param_name' => 'columns',
							'value' => array(
								'' => '',
								esc_html__('One','mikado-cpt') => '1',
								esc_html__('Two','mikado-cpt') => '2',
								esc_html__('Three','mikado-cpt') => '3',
								esc_html__('Four','mikado-cpt') => '4',
								esc_html__('Five','mikado-cpt') => '5',
								esc_html__('Six','mikado-cpt') => '6'
							),
							'description' => esc_html__('Default value is Three','mikado-cpt'),
							'dependency' => array('element' => 'type', 'value' => array('standard','gallery','gallery-with-space')),
							'group' => esc_html__('Query and Layout Options','mikado-cpt')
						),
                        array(
                            'type' => 'dropdown',
                            'heading' => esc_html__('Number of Columns for Masonry','mikado-cpt'),
                            'param_name' => 'masonry_columns',
                            'value' => array(
                                '' => '',
                                esc_html__('Three','mikado-cpt') => '3',
                                esc_html__('Four','mikado-cpt') => '4',
                            ),
                            'description' => esc_html__('Default value is Four','mikado-cpt'),
                            'dependency' => array('element' => 'type', 'value' => array('masonry-with-space','masonry')),
                            'group' => esc_html__('Query and Layout Options','mikado-cpt')
                        ),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Grid Size','mikado-cpt'),
							'param_name' => 'grid_size',
							'value' => array(
								esc_html__('Default','mikado-cpt') => '',
								esc_html__('3 Columns Grid','mikado-cpt') => 'three',
								esc_html__('4 Columns Grid','mikado-cpt') => 'four',
								esc_html__('5 Columns Grid','mikado-cpt') => 'five'
							),
							'dependency' => array('element' => 'type', 'value' => array('pinterest','pinterest-with-space')),
							'group' => esc_html__('Query and Layout Options','mikado-cpt')
						),
						array(
							'type' => 'textfield',
							'heading' => esc_html__('Show Only Projects with Listed IDs','mikado-cpt'),
							'param_name' => 'selected_projects',
							'description' => esc_html__('Delimit ID numbers by comma (leave empty for all)','mikado-cpt'),
							'group' => esc_html__('Query and Layout Options','mikado-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Enable Category Filter','mikado-cpt'),
							'param_name' => 'filter',
							'value' => array(
								esc_html__('No','mikado-cpt') => 'no',
								esc_html__('Yes','mikado-cpt') => 'yes'
							),
							'save_always' => true,
							'description' => esc_html__('Default value is No','mikado-cpt'),
							'group' => esc_html__('Query and Layout Options','mikado-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Filter Order By','mikado-cpt'),
							'param_name' => 'filter_order_by',
							'value' => array(
								esc_html__('Name','mikado-cpt')  => 'name',
								esc_html__('Count','mikado-cpt') => 'count',
								esc_html__('Id','mikado-cpt')    => 'id',
								esc_html__('Slug','mikado-cpt')  => 'slug'
							),
							'save_always' => true,
							'description' => esc_html__('Default value is Name','mikado-cpt'),
							'dependency' => array('element' => 'filter', 'value' => array('yes')),
							'group' => esc_html__('Query and Layout Options','mikado-cpt')
						)
					)
				)
			);
		}
	}

	/**
	 * Renders shortcodes HTML
	 *
	 * @param $atts array of shortcode params
	 * @param $content string shortcode content
	 * @return string
	 */
	public function render($atts, $content = null) {

		$args = array(
			'type' => 'standard',
			'columns' => '3',
            'masonry_columns' => '4',
			'grid_size' => 'three',
			'image_size' => 'full',
			'hover_type' => 'zoom-out-simple',
			'order_by' => 'date',
			'order' => 'ASC',
			'number' => '-1',
			'filter' => 'no',
			'filter_order_by' => 'name',
			'category' => '',
			'selected_projects' => '',
			'show_more' => 'load_more',
			'title_tag' => 'h3',
			'next_page' => '',
			'portfolio_slider' => '',
			'portfolios_shown' => ''
		);

		$params = shortcode_atts($args, $atts);
		extract($params);

		$query_array = $this->getQueryArray($params);
		$query_results = new \WP_Query($query_array);
		$params['query_results'] = $query_results;
		$params['masonry_filter'] = '';
		$params['with_space'] = false;

		//check if any with space type is selected, if so, changes the type to without space and changes with_space attribute
		if (mkd_core_string_ends_with($params['type'],'-with-space')){
			$params['type'] = mkd_core_filter_suffix($params['type'],'-with-space');
			$type = $params['type'];
			$params['with_space'] = true;
		}

		$classes = $this->getPortfolioClasses($params);
		$data_atts = $this->getDataAtts($params);
		$data_atts .= 'data-max-num-pages = '.$query_results->max_num_pages;

		$html = '';


		if($filter == 'yes' && ($type == 'masonry' || $type =='pinterest')){
			$params['filter_categories'] = $this->getFilterCategories($params);
			$params['masonry_filter'] = 'mkdf-masonry-filter';
			$html .= mkd_core_get_shortcode_module_template_part('portfolio','portfolio-filter', '', $params);
		}

		$html .= '<div class = "mkdf-portfolio-list-holder-outer '.$classes.'" '.$data_atts. '>';

		if($filter == 'yes' && ($type == 'standard' || $type =='gallery')){
			$params['filter_categories'] = $this->getFilterCategories($params);
			$html .= mkd_core_get_shortcode_module_template_part('portfolio','portfolio-filter', '', $params);
		}

		$html .= '<div class = "mkdf-portfolio-list-holder clearfix" >';
		if($type == 'masonry' || $type == 'pinterest'){
			$html .= '<div class="mkdf-portfolio-list-masonry-grid-sizer"></div>';
			$html .= '<div class="mkdf-portfolio-list-masonry-grid-gutter"></div>';
		}

		if($query_results->have_posts()):
			while ( $query_results->have_posts() ) : $query_results->the_post();

				$params['current_id'] = get_the_ID();
				$params['thumb_size'] = $this->getImageSize($params);
				$params['category_html'] = $this->getItemCategoriesHtml($params);
				$params['categories'] = $this->getItemCategories($params);
				$params['article_masonry_size'] = $this->getMasonrySize($params);
				$params['item_link'] = $this->getItemLink($params);
				$params['hover_params']['category_html'] = $params['category_html'];
				$params['hover_params']['title_tag'] = $params['title_tag'];
				$params['hover_params']['item_link'] = $params['item_link'];

				$html .= mkd_core_get_shortcode_module_template_part('portfolio',$type, '', $params);

			endwhile;
		else:

			$html .= '<p>'. esc_html__( 'Sorry, no posts matched your criteria.' ) .'</p>';

		endif;
		if(($type =='gallery' || $type == 'standard') && $portfolio_slider !== 'yes'){
			for($i=0;$i<(int)$columns;$i++){
				$html .= "<div class='mkdf-portfolio-gap'></div>\n";
			}
		}

		$html .= '</div>'; //close mkdf-portfolio-list-holder
		if($show_more !== 'none'){
			$html .= mkd_core_get_shortcode_module_template_part('portfolio','load-more-template', '', $params);
		}
		wp_reset_postdata();
		$html .= '</div>'; // close mkdf-portfolio-list-holder-outer
		return $html;
	}

	/**
	 * Generates portfolio list query attribute array
	 *
	 * @param $params
	 *
	 * @return array
	 */
	public function getQueryArray($params){

		$query_array = array();

		$query_array = array(
			'post_type' => 'portfolio-item',
			'orderby' =>$params['order_by'],
			'order' => $params['order'],
			'posts_per_page' => $params['number']
		);

		if(!empty($params['category'])){
			$query_array['portfolio-category'] = $params['category'];
		}

		$project_ids = null;
		if (!empty($params['selected_projects'])) {
			$project_ids = explode(',', $params['selected_projects']);
			$query_array['post__in'] = $project_ids;
		}

		$paged = '';
		if(empty($params['next_page'])) {
			if(get_query_var('paged')) {
				$paged = get_query_var('paged');
			} elseif(get_query_var('page')) {
				$paged = get_query_var('page');
			}
		}

		if(!empty($params['next_page'])){
			$query_array['paged'] = $params['next_page'];

		}else{
			$query_array['paged'] = 1;
		}

		return $query_array;
	}

	/**
	 * Generates portfolio classes
	 *
	 * @param $params
	 *
	 * @return string
	 */
	public function getPortfolioClasses($params){
		$classes = array();
		$type = $params['type'];
		$with_space = $params['with_space'];
		$hover_type = $params['hover_type'];
		$columns = $params['columns'];
        $masonry_columns = $params['masonry_columns'];
		$grid_size = $params['grid_size'];
		switch($type):
			case 'standard':
				$classes[] = 'mkdf-ptf-standard';
				break;
			case 'gallery':
				$classes[] = 'mkdf-ptf-gallery';
				break;
			case 'masonry':
				$classes[] = 'mkdf-ptf-masonry';
				break;
			case 'pinterest':
				$classes[] = 'mkdf-ptf-pinterest';
				break;
		endswitch;

		if ($with_space){
			$classes[] = 'mkdf-ptf-with-space';
		}

		if ($hover_type !== '' && $type !== 'standard'){
			$classes[] = 'mkdf-ptf-hover-'.$hover_type;
		}

		if(empty($params['portfolio_slider'])){ // portfolio slider mustn't have this classes

			if($type == 'standard' || $type == 'gallery'){
				switch ($columns):
					case '1':
						$classes[] = 'mkdf-ptf-one-column';
						break;
					case '2':
						$classes[] = 'mkdf-ptf-two-columns';
						break;
					case '3':
						$classes[] = 'mkdf-ptf-three-columns';
						break;
					case '4':
						$classes[] = 'mkdf-ptf-four-columns';
						break;
					case '5':
						$classes[] = 'mkdf-ptf-five-columns';
						break;
					case '6':
						$classes[] = 'mkdf-ptf-six-columns';
						break;
				endswitch;
			}
            if($type == 'masonry' || $type == 'masonry-with-space'){
                switch ($masonry_columns):
                    case '3':
                        $classes[] = 'mkdf-ptfm-three-columns';
                        break;
                    case '4':
                        $classes[] = 'mkdf-ptfm-four-columns';
                        break;
                endswitch;
            }
			if($params['show_more'] == 'load_more'){
				$classes[] = 'mkdf-ptf-load-more mkdf-ptf-show-more';
			}
			else if($params['show_more'] == 'infinite_scroll'){
				$classes[] = 'mkdf-ptf-infinite-scroll mkdf-ptf-show-more';
			}

		}

		if($type == "pinterest"){
			switch ($grid_size):
				case 'three':
					$classes[] = 'mkdf-ptf-pinterest-three-columns';
					break;
				case 'four':
					$classes[] = 'mkdf-ptf-pinterest-four-columns';
					break;
				case 'five':
					$classes[] = 'mkdf-ptf-pinterest-five-columns';
					break;
			endswitch;
		}
		if($params['filter'] == 'yes'){
			$classes[] = 'mkdf-ptf-has-filter';
			if($params['type'] == 'masonry' || $params['type'] == 'pinterest'){
				if($params['filter'] == 'yes'){
					$classes[] = 'mkdf-ptf-masonry-filter';
				}
			}
		}

		if(!empty($params['portfolio_slider']) && $params['portfolio_slider'] == 'yes'){
			$classes[] = 'mkdf-portfolio-slider-holder';
		}

		return implode(' ',$classes);

	}
	/**
	 * Generates portfolio image size
	 *
	 * @param $params
	 *
	 * @return string
	 */
	public function getImageSize($params){

		$thumb_size = 'full';
		$type = $params['type'];

		if($type == 'standard' || $type == 'gallery'){
			if(!empty($params['image_size'])){
				$image_size = $params['image_size'];

				switch ($image_size) {
					case 'landscape':
						$thumb_size = 'cortex_mikado_landscape';
						break;
					case 'portrait':
						$thumb_size = 'cortex_mikado_portrait';
						break;
					case 'square':
						$thumb_size = 'cortex_mikado_square';
						break;
					case 'full':
						$thumb_size = 'full';
						break;
				}
			}
		}
		elseif($type == 'masonry'){

			$id = $params['current_id'];
			$masonry_size = get_post_meta($id, 'portfolio_masonry_dimenisions',true);

			switch($masonry_size):
				default :
					$thumb_size = 'cortex_mikado_square';
					break;
				case 'large_width' :
					$thumb_size = 'cortex_mikado_large_width';
					break;
				case 'large_height' :
					$thumb_size = 'cortex_mikado_large_height';
					break;
				case 'large_width_height' :
					$thumb_size = 'cortex_mikado_large_width_height';
					break;
			endswitch;
		}


		return $thumb_size;
	}
	/**
	 * Generates portfolio item categories ids.This function is used for filtering
	 *
	 * @param $params
	 *
	 * @return array
	 */
	public function getItemCategories($params){
		$id = $params['current_id'];
		$category_return_array = array();

		$categories = wp_get_post_terms($id, 'portfolio-category');

		foreach($categories as $cat){
			$category_return_array[] = 'portfolio_category_'.$cat->term_id;
		}
		return implode(' ', $category_return_array);
	}

	/**
	 * Generates portfolio item categories html based on id
	 *
	 * @param $params
	 *
	 * @return html
	 */
	public function getItemCategoriesHtml($params){
		$id = $params['current_id'];

		$categories = wp_get_post_terms($id, 'portfolio-category');
		$category_html = '<div class="mkdf-ptf-category-holder">';
		$k = 1;

		foreach ($categories as $cat) {
			$category_html .= '<span>'.$cat->name.'</span>';
			if (count($categories) != $k) {
				$category_html .= ' , ';
			}
			$k++;
		}
		$category_html .= '</div>';
		return $category_html;
	}

	/**
	 * Generates masonry size class for each article( based on id)
	 *
	 * @param $params
	 *
	 * @return string
	 */
	public function getMasonrySize($params){
		$masonry_size_class = '';

		if($params['type'] == 'masonry'){

			$id = $params['current_id'];
			$masonry_size = get_post_meta($id, 'portfolio_masonry_dimenisions',true);
			switch($masonry_size):
				default :
					$masonry_size_class = 'mkdf-default-masonry-item';
					break;
				case 'large_width' :
					$masonry_size_class = 'mkdf-large-width-masonry-item';
					break;
				case 'large_height' :
					$masonry_size_class = 'mkdf-large-height-masonry-item';
					break;
				case 'large_width_height' :
					$masonry_size_class = 'mkdf-large-width-height-masonry-item';
					break;
			endswitch;
		}

		return $masonry_size_class;
	}
	/**
	 * Generates filter categories array
	 *
	 * @param $params
	 *
	 * * @return array
	 */
	public function getFilterCategories($params){

		$cat_id = 0;
		$top_category = '';

		if(!empty($params['category'])){

			$top_category = get_term_by('slug', $params['category'], 'portfolio-category');
			if(isset($top_category->term_id)){
				$cat_id = $top_category->term_id;
			}

		}

		$args = array(
			'taxonomy' => 'portfolio-category',
			'child_of' => $cat_id,
			'order_by' => $params['filter_order_by']
		);

		$filter_categories = get_terms($args);

		return $filter_categories;

	}
	/**
	 * Generates datta attributes array
	 *
	 * @param $params
	 *
	 * @return array
	 */
	public function getDataAtts($params){

		$data_attr = array();
		$data_return_string = '';

		if(get_query_var('paged')) {
			$paged = get_query_var('paged');
		} elseif(get_query_var('page')) {
			$paged = get_query_var('page');
		} else {
			$paged = 1;
		}

		if(!empty($paged)) {
			$data_attr['data-next-page'] = $paged+1;
		}
		if(!empty($params['type'])){
			$data_attr['data-type'] = $params['type'];
		}
		if(!empty($params['columns'])){
			$data_attr['data-columns'] = $params['columns'];
		}
		if(!empty($params['grid_size'])){
			$data_attr['data-grid-size'] = $params['grid_size'];
		}
		if(!empty($params['order_by'])){
			$data_attr['data-order-by'] = $params['order_by'];
		}
		if(!empty($params['order'])){
			$data_attr['data-order'] = $params['order'];
		}
		if(!empty($params['number'])){
			$data_attr['data-number'] = $params['number'];
		}
		if(!empty($params['image_size'])){
			$data_attr['data-image-size'] = $params['image_size'];
		}
		if(!empty($params['hover_type'])){
			$data_attr['data-hover-type'] = $params['hover_type'];
		}
		if(!empty($params['filter'])){
			$data_attr['data-filter'] = $params['filter'];
		}
		if(!empty($params['filter_order_by'])){
			$data_attr['data-filter-order-by'] = $params['filter_order_by'];
		}
		if(!empty($params['category'])){
			$data_attr['data-category'] = $params['category'];
		}
		if(!empty($params['selected_projects'])){
			$data_attr['data-selected-projects'] = $params['selected_projects'];
		}
		if(!empty($params['show_more'])){
			$data_attr['data-show-more'] = $params['show_more'];
		}
		if(!empty($params['title_tag'])){
			$data_attr['data-title-tag'] = $params['title_tag'];
		}
		if(!empty($params['portfolio_slider']) && $params['portfolio_slider']=='yes'){
			$data_attr['data-items'] = $params['portfolios_shown'];
		}

		foreach($data_attr as $key => $value) {
			if($key !== '') {
				$data_return_string .= $key . '= "' . esc_attr( $value ) . '" ';
			}
		}
		return $data_return_string;
	}

	public function getItemLink($params){

		$id = $params['current_id'];
		$portfolio_link = get_permalink($id);
		if (get_post_meta($id, 'portfolio_external_link',true) !== ''){
			$portfolio_link = get_post_meta($id, 'portfolio_external_link',true);
		}

		return $portfolio_link;

	}
}